//=============================================================================
// ExtraCritical.js
// ----------------------------------------------------------------------------
// (C)2022 kuroame koubou
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 0.9.0 2022/07/17 初版
// ----------------------------------------------------------------------------
// [Blog]   : http://kuroamekoubou.blog.fc2.com
// [Twitter]: https://twitter.com/kuroVelter9623
//=============================================================================

/*:
 * @plugindesc 多段クリティカル機構プラグイン
 * @target MZ
 * @author くろあめ
 * 
 * @param Crt Str Lv1
 * @text 第一クリティカル表記
 * @desc 第一クリティカル成功時のメッセージです。
 * @default クリティカル！
 * @type text
 * 
 * @param Crt Str Lv2
 * @text 第二クリティカル表記
 * @desc 第二クリティカル成功時のメッセージです。
 * @default へビィクリティカル！
 * @type text
 * 
 * @param Crt Str Lv3
 * @text 第三クリティカル表記
 * @desc 第三クリティカル成功時のメッセージです。
 * @default バスタークリティカル！
 * @type text
 *
 * @param Critical Max
 * @text クリティカル最大段階
 * @desc クリティカルの最大判定回数です。上限は３です。
 * @default 3
 * @type number
 *
 * @param Base Crt Rate
 * @text クリティカル基礎倍率
 * @desc クリティカル時の基礎倍率(%)です。クリティカル段階が1の時の倍率です。
 * @default 150
 * @type number
 * 
 * @param Plus Crt Rate
 * @text クリティカル加算倍率
 * @desc 2段階以降の加算倍率(%)です。クリティカル段階が2以上になると、段階毎に、この倍率が加算されます。
 * @default 50
 * @type number
 * 
 * @param Lower Crt Rate
 * @text クリティカル倍率最低保証値
 * @desc クリティカル倍率減殺補正発動時の、最低保証クリティカル倍率です。減殺補正があってもこれ以下にはなりません。
 * @default 125
 * @type number
 *
 * @param Prob Crt LuckRate
 * @text 運気補正確率
 * @desc 運気の差で、クリティカル率に掛かる補正(%)です。行動者の運気が相手の運気の2倍のとき、この補正が加算されます。（相手より運気が低い場合、マイナス補正はかかりません）
 * @default 0.25
 * @type number
 * 
 * @param Prob Crt LuckMax
 * @text 運気補正確率限界閾値
 * @desc 運気の差で、クリティカル率が変動する最大の差分倍率です。初期設定の3なら3倍以上の差は無視されます。
 * @default 3
 * @type number
 * 
 * @param Prob Crt DamRate
 * @text 運気補正倍率
 * @desc 運気の差で、クリティカル倍率に掛かる補正(%)です。行動者と相手に運気に2倍の差がついているとき、この倍率が±されます。
 * @default 0.20
 * @type number
 * 
 * @param Prob Crt DamMax
 * @text 運気補正倍率限界閾値
 * @desc 運気の差で、クリティカル倍率が変動する最大の差分倍率です。初期設定の3なら3倍以上の差は無視されます。
 * @default 4
 * @type number
 *
 * @help
 * クリティカル発生時に、クリティカル判定を複数回行い
 * 判定に成功する程、威力が上昇する
 * 多段クリティカルシステムを導入します
 * 最大三段階までのクリティカルレベルの導入が可能です。
 * 
 * クリティカルの発生率は、クリティカルレベルが1上がるごとに
 * 本来のクリティカル率の、1/2、1/3と下がっていきます
 * 一番最初の判定の確率は、攻撃者とスキルの総合改心率そのものです。
 * 
 * これに加え、スキルのメモ欄に<CriProbBst:XX>と記載することで
 * そのスキルのクリティカル発生率をXX%上げることができます。
 * 
 * また、スキルのメモ欄に<CriLevelFix:X>（Xは1~3）と記載することで
 * そのスキルの使用時に、X段階のクリティカルを確定発生させることができます。
 * 
 * 同時に、クリティカル時の補正倍率
 * 「必殺倍率」を、スキルやキャラクターごとに
 * 設定できるようになります
 * 必殺倍率はクリティカル成功時の基礎倍率に加算され
 * 多段判定成功時の倍率加算時には加味されません。
 * 
 * アクター、エネミー、職業、装備、スキルのメモ欄に
 * <CriMagniBst:XX> と記載すると、攻撃の必殺倍率がXX%増加します。
 *
 * 【設定例(本プラグインの初期設定、基本必殺倍率150%基準)】
 * アクターのメモ欄に<CriMagniBst:10> と記載した場合
 * そのアクターの必殺倍率は160%になります。
 *
 * メモ欄に<CriMagniBst:15> と記載された武器を装備したアクターが
 * メモ欄に<CriMagniBst:30> と記載されたスキルで攻撃した場合
 * そのスキルの必殺倍率は195%になります。
 *
 * 同じく、アクター、エネミー、職業、装備、スキルのメモ欄に
 * <CriMagniReg:XX> と記載すると、攻撃の必殺倍率がXX%減少します。
 * 
 * ルールは必殺倍率増加のタグと一緒ですが
 * こちらはプラグインの設定で設けた下限倍率以下にはなりません。
 * 
 * 多段クリティカル判定は敵味方共通です
 * クリティカル表示文字も統一されます.
  *
 * このプラグインは、戦闘計算で使われる各種関数を再定義する都合上
 * 非常に競合性の高い作りになっています
 * ほかの戦闘計算式系スクリプトと共存させるときには
 * 細心の注意を払ってください。
 *
 *
 * 利用形態（商用、18禁利用等）について制限はありません
 * 作者への使用報告や、ReadMeなどへ作者・サイト名の記載も任意となります。
 *
 * ただし、作者はこれらのプラグインについて
 * 基本的に一切のサポートを行いません
 * 何か問題が起きた時、ご自身で解決できる方のみご利用ください。
 */


(function() {

const parameters     = PluginManager.parameters('ExtraCritical');
const crtMax         = Number(parameters['Critical Max']);
const crtBaseRate    = Number(parameters['Base Crt Rate']);
const crtPlusRate    = Number(parameters['Plus Crt Rate']);
const crtLowerRate   = Number(parameters['Lower Crt Rate']);
const ProbCrtLuckRate = Number(parameters['Prob Crt LuckRate']);
const ProbCrtLuckMax  = Number(parameters['Prob Crt LuckMax']);
const ProbCrtDamRate = Number(parameters['Prob Crt DamRate']);
const ProbCrtDamMax  = Number(parameters['Prob Crt DamMax']);
const crtStrLv1      = String(parameters['Crt Str Lv1'] || 'クリティカル！');
const crtStrLv2      = String(parameters['Crt Str Lv2'] || 'へビィクリティカル！');
const crtStrLv3      = String(parameters['Crt Str Lv3'] || 'バスタークリティカル！');

/* アクターのクリティカル倍率増幅値取得 */
Game_Actor.prototype.getCriMagniBst = function() {
    
    var fullMagni = 0;

    var states = this.states();
    for (let i = 0; i < states.length; i++) {
        if (states[i] && states[i].meta && states[i].meta.CriMagniBst) fullMagni += Number(states[i].meta.CriMagniBst);
    }
     
    if (this.actor() && this.actor().meta && this.actor().meta.CriMagniBst) fullMagni += Number(this.actor().meta.CriMagniBst);
    if (this.currentClass() && this.currentClass().meta && this.currentClass().meta.CriMagniBst) fullMagni += Number(this.currentClass().meta.CriMagniBst);
    var equips = this.equips();
    for (let i = 0; i < equips.length; i++) {
        if (equips[i] && equips[i].meta && equips[i].meta.CriMagniBst) fullMagni += Number(equips[i].meta.CriMagniBst);
    }
    
    return fullMagni;
};

/* アクターの表示用クリティカル倍率増幅値取得 */
Game_Actor.prototype.dispCriMagni = function() {
    return crtBaseRate + this.getCriMagniBst();
};

/* アクターのクリティカル倍率減殺値取得 */
Game_Actor.prototype.getCriMagniReg = function() {

    var regMagni = 0;
    
    var states = this.states();
    for (let i = 0; i < states.length; i++) {
        if (states[i] && states[i].meta && states[i].meta.CriMagniReg) regMagni -= Number(states[i].meta.CriMagniReg);
    }
    
    if (this.actor() && this.actor().meta && this.actor().meta.CriMagniReg) regMagni -= Number(this.actor().meta.CriMagniReg);
    if (this.currentClass() && this.currentClass().meta && this.currentClass().meta.CriMagniReg) regMagni -= Number(this.currentClass().meta.CriMagniReg);
    var equips = this.equips();
    for (let i = 0; i < equips.length; i++) {
        if (equips[i] && equips[i].meta && equips[i].meta.CriMagniReg) regMagni -= Number(equips[i].meta.CriMagniReg);
    }
    
    return regMagni;
};

/* アクターの表示用クリティカル倍率減殺値取得 */
Game_Actor.prototype.dispCriMagniReg = function() {
    return this.getCriMagniReg();
};

/* エネミーのクリティカル倍率増幅値取得 */
Game_Enemy.prototype.getCriMagniBst = function() {
    
    var fullMagni = 0;
    
    var states = this.states();
    for (let i = 0; i < states.length; i++) {
        if (states[i] && states[i].meta && states[i].meta.CriMagniBst) fullMagni += Number(states[i].meta.CriMagniBst);
    }
     
    if (this.enemy() && this.enemy().meta && this.enemy().meta.CriMagniBst) fullMagni += Number(this.enemy().meta.CriMagniBst);
    
    return fullMagni;
};

/* エネミーのクリティカル倍率減殺値取得 */
Game_Enemy.prototype.getCriMagniReg = function() {

    var regMagni = 0;
    
    var states = this.states();
    for (let i = 0; i < states.length; i++) {
        if (states[i] && states[i].meta && states[i].meta.CriMagniReg) regMagni -= Number(states[i].meta.CriMagniReg);
    }
    if (this.enemy() && this.enemy().meta && this.enemy().meta.CriMagniReg) regMagni -= Number(this.enemy().meta.CriMagniReg);
    
    return regMagni;
};

/* 表示用クリティカル倍率増幅値取得(プロパティ) */
Object.defineProperties(Game_BattlerBase.prototype, {
    crr: {
        get: function() {
            return crtBaseRate + this.getCriMagniBst();
        },
        configurable: true
    }
});

/* 攻撃者の全必殺倍率を合算して返す */
Game_Action.prototype.fullCriMagni = function(attacker,target,item,critical_lv ) {
    
    var fullMagni = 0;
    
    if (item && item.meta && item.meta.CriMagniBst) fullMagni += Number(item.meta.CriMagniBst);
    fullMagni += attacker.getCriMagniBst();
    
    if(fullMagni > 0 && critical_lv > 1){
        fullMagni += Math.round(fullMagni * 0.25 * (critical_lv - 1));
    }
    
    fullMagni += target.getCriMagniReg();
    
    return fullMagni;
};

/* 都合上、元処理は退避するだけで呼び出さない 処理ごと再定義 */
const _Game_Action_itemCri_og = Game_Action.prototype.itemCri;
Game_Action.prototype.itemCri = function(target,cricnt) {
    var criProbPlus = 1.0;
    if(this.subject().luk >= target.luk){
       criProbPlus = 1.0 + (ProbCrtLuckRate * Math.min(this.subject().luk / target.luk, ProbCrtLuckMax));
    }
    var fullCriRate = this.subject().cri;
    fullCriRate += this.item().meta.CriProbBst ? this.item().meta.CriProbBst / 100.0 : 0;
    
    return this.item().damage.critical ? (fullCriRate * criProbPlus / cricnt) * (1 - target.cev) : 0;
};


/* 都合上、元処理は退避するだけで呼び出さない 処理ごと再定義 */
const _Game_Action_apply_og = Game_Action.prototype.apply;
Game_Action.prototype.apply = function(target) {
    const result = target.result();
    this.subject().clearResult();
    result.clear();
    result.used = this.testApply(target);
    result.missed = result.used && Math.random() >= this.itemHit(target);
    result.evaded = !result.missed && Math.random() < this.itemEva(target);
    result.physical = this.isPhysical();
    result.drain = this.isDrain();
    if (result.isHit()) {
        if (this.item().damage.type > 0) {
/* ここから処理変更 */
            result.critical = false;
            result.critical_lv = 0;
            
            var cricnt=1;
            if (this.item().meta.CriLevelFix){
                cricnt=Number(this.item().meta.CriLevelFix);
                result.critical = true;
                result.critical_lv = cricnt;
            }
            if (cricnt>=crtMax){
                cricnt=crtMax
                result.critical_lv = cricnt;
            }
            
            for(cricnt; cricnt<=crtMax; cricnt++){
                if(Math.random() < this.itemCri(target,cricnt)){
                    result.critical = true;
                    result.critical_lv = cricnt;
                }else{
                    break;
                }
            }
/* ここまで処理変更 */
            const value = this.makeDamageValue(target, result.critical);
            this.executeDamage(target, value);
        }
        for (const effect of this.item().effects) {
            this.applyItemEffect(target, effect);
        }
        this.applyItemUserEffect(target);
    }
    this.updateLastTarget(target);
};

/* 都合上、元処理は退避するだけで呼び出さない 処理ごと再定義 */
const _Game_Action_makeDamageValue_og = Game_Action.prototype.makeDamageValue;
Game_Action.prototype.makeDamageValue = function(target, critical) {
    const result = target.result();
    const item = this.item();
    const baseValue = this.evalDamageFormula(target);
    result.elementRate = this.calcElementRate(target);
    let value = baseValue * this.calcElementRate(target);
    if (this.isPhysical()) {
        value *= target.pdr;
    }
    if (this.isMagical()) {
        value *= target.mdr;
    }
    if (baseValue < 0) {
        value *= target.rec;
    }
    if (critical) {
        value = this.applyCritical(value,target);
    }
    value = this.applyVariance(value, item.damage.variance);
    value = this.applyGuard(value, target);
    value = Math.round(value);
    return value;
};


/* 都合上、元処理は退避するだけで呼び出さない 処理ごと再定義 */
const _Game_Action_applyCritical_og = Game_Action.prototype.applyCritical;
Game_Action.prototype.applyCritical = function(damage,target) {
    var result = target.result();
    var lastMagni = crtBaseRate + (crtPlusRate * (result.critical_lv -1));
    lastMagni += this.fullCriMagni(this.subject(),target,this.item(),result.critical_lv);
    var criProbPlus = 1.0;
    if(this.subject().luk >= target.luk){
       criProbPlus = 1.0 + (ProbCrtDamRate * Math.min(this.subject().luk / target.luk, ProbCrtDamMax));
    }
    lastMagni *= criProbPlus;
    if (lastMagni < crtLowerRate) lastMagni = crtLowerRate;
    
    return damage * lastMagni / 100;
};

/* クリティカルレベルの初期化処理追加 */
const _Game_Action_clear_excri = Game_ActionResult.prototype.clear;
Game_ActionResult.prototype.clear = function() {
    _Game_Action_clear_excri.apply(this, arguments);
    this.critical_lv = 0;
};

/* 都合上、元処理は退避するだけで呼び出さない 処理ごと再定義 */
const _Window_BattleLog_displayCritical_og = Window_BattleLog.prototype.displayCritical;
    Window_BattleLog.prototype.displayCritical = function(target) {
        
        var pushText;
        pushText = "";
        
        if (target.result().critical) {
            if(target.result().critical_lv >= 3) {
                $gameScreen.startShake(3, 20, 10); // 仮仕様 将来的にはパラメータ化 
                pushText = crtStrLv3;
            }
            else if(target.result().critical_lv >= 2) {
                $gameScreen.startShake(2.5, 18, 10); // 仮仕様 将来的にはパラメータ化 
                pushText = crtStrLv2;
            }
            else{
                pushText = crtStrLv1;
            }
        }
        if(target.result().hpDamage > 0 && target.result().elementRate && target.result().elementRate >= 1.01){
            pushText += " 弱点属性！";
        }
        if(target.result().hpDamage > 0 && target.result().elementRate && target.result().elementRate <= 0.99){
            pushText += " 軽減属性！";
        }
        if(pushText != ""){
            this.push('addText', pushText);
        }
    };


/*
Window_BattleLog.prototype.displayCritical = function(target) {
    if (target.result().critical) {
        if(target.result().critical_lv >= 3) {
            $gameScreen.startShake(3, 20, 10); // 仮仕様 将来的にはパラメータ化 
            this.push('addText', crtStrLv3);
        }
        else if(target.result().critical_lv >= 2) {
            $gameScreen.startShake(2.5, 18, 10); // 仮仕様 将来的にはパラメータ化 
            this.push('addText', crtStrLv2);
        }
        else{
            this.push('addText', crtStrLv1);
        }
    }
};
*/

})();

